#include "StdV3D.h"
#include "BlobEllipsoid.h"

namespace V3D {


BlobEllipsoid::BlobEllipsoid(float fCenterEnergy, const Vector3D& vcPos, float fInfluenceRange)
: parent(fCenterEnergy, vcPos, fInfluenceRange) 
{
	NotifyBoundsChange();
}

BlobEllipsoid::BlobEllipsoid(float fCenterEnergy /* = 1.f */)
: parent(fCenterEnergy, Vector3D(0,0,0), 1.f) 
{
	NotifyBoundsChange();
}

BlobEllipsoid::~BlobEllipsoid()
{}


bool BlobEllipsoid::ComputeBlobBounds(Vector3D& vcBallMin, Vector3D& vcBallMax) const
{
	const MatrixTransf& matTransfBaseShape    = GetTransform();
	const MatrixTransf& matTransfBaseShapeInv = GetInverseTransform();

	Vector3D vcSphereRectBaseX; vcSphereRectBaseX.Set( matTransfBaseShapeInv.GetColumnX() );
	Vector3D vcSphereRectBaseY; vcSphereRectBaseY.Set( matTransfBaseShapeInv.GetColumnY() );
	Vector3D vcSphereRectBaseZ; vcSphereRectBaseZ.Set( matTransfBaseShapeInv.GetColumnZ() );


//	float fTmp = m_fInfluenceRange / Abs(( vcSphereRectBaseX ^ vcSphereRectBaseY) * vcSphereRectBaseZ);
//	float fTmp = 1.f / Abs(( vcSphereRectBaseX ^ vcSphereRectBaseY) * vcSphereRectBaseZ);
	LengthType rTmp = LengthType( fabs( matTransfBaseShape.GetDeterminant() ) );
	LengthType rHalfLengthX = rTmp * (vcSphereRectBaseY ^ vcSphereRectBaseZ).GetLength();
	LengthType rHalfLengthY = rTmp * (vcSphereRectBaseX ^ vcSphereRectBaseZ).GetLength();
	LengthType rHalfLengthZ = rTmp * (vcSphereRectBaseX ^ vcSphereRectBaseY).GetLength();

	assert( rHalfLengthX >= 0 && rHalfLengthY >= 0 && rHalfLengthZ >= 0);
	const Vector3D vcBoundBallHalfSize = Vector3D( rHalfLengthX, rHalfLengthY, rHalfLengthZ);

	const Vector3D& vcCenter = matTransfBaseShape.GetTranslation();

	vcBallMin = vcCenter - vcBoundBallHalfSize;
	vcBallMax = vcCenter + vcBoundBallHalfSize;
	return true;
}



/*
// Calcule a partir de la boite definie par vcBoxMin et vcBoxMax, son sous-ensemble minimum contenant le blob
// Retourne true si cette boite minimum existe (volume non nul)
bool BlobEllipsoid::ComputeMinEnclosingBoxAligned(Vector3D& vcMin, Vector3D& vcMax, const Vector3D& vcBoxMin, const Vector3D& vcBoxMax) const
{
	return parent::ComputeMinEnclosingBoxAligned( vcMin, vcMax, vcBoxMin, vcBoxMax);

	Vector3D vcBoxTransfPos = m_matTransfBaseShapeInv * vcBoxMin;

	Vector3D vcBoxTransfX, vcBoxTransfY, vcBoxTransfZ;
	m_matTransfBaseShapeInv.GetColumns(vcBoxTransfX, vcBoxTransfY, vcBoxTransfZ);

	Vector3D vcBoxDiag = vcBoxMax; vcBoxDiag -= vcBoxMin;
	vcBoxTransfX *= vcBoxDiag.x;
	vcBoxTransfY *= vcBoxDiag.y;
	vcBoxTransfZ *= vcBoxDiag.z;

	// La matrice (vcBoxTransfX, vcBoxTransfY, vcBoxTransfZ, vcBoxTransfPos) decrit le cube
	// dans la base ou l'ellipsoide est une sphere de rayon 1, centree en O
	

	// Pour la class BlobBase, on ne connait rien de plus que la boite englobante alignee sur les axes, 
	// La boite minimum est l'intersection de cette boite englobante avec la boite fournie en parametres.
	GetBlobBounds(vcMin, vcMax);

	if( vcMin.x < vcBoxMin.x) vcMin.x = vcBoxMin.x;
	if( vcMin.y < vcBoxMin.y) vcMin.y = vcBoxMin.y;
	if( vcMin.z < vcBoxMin.z) vcMin.z = vcBoxMin.z;

	if( vcMax.x > vcBoxMax.x) vcMax.x = vcBoxMax.x;
	if( vcMax.y > vcBoxMax.y) vcMax.y = vcBoxMax.y;
	if( vcMax.z > vcBoxMax.z) vcMax.z = vcBoxMax.z;

	// Optimisation : operateur | nous assure que si une des conditions est vraie, l'expression est vraie,
	//                l'operateur & ne nous donne pas la meme garantie 
	bool bVolumeNoExists = bool( (vcMax.x - vcMin.x <= 0.f) |
								 (vcMax.y - vcMin.y <= 0.f) |
								 (vcMax.z - vcMin.z <= 0.f) );

	return !bVolumeNoExists;
}
*/




void BlobEllipsoid::EnergyBaseShapeAddAtRectangle ( float* pafAddEnergies,    // Tableau des valeurs energetiques de la grille
										 const Vector3D& vcRectStart,  // Origine du rectangle
										 const Vector3D& vcRectAxisX,  // Axe X du rectangle (sert d'unite de separation entre deux calculs d'energie)
										 const Vector3D& vcRectAxisY,  // Axe Y du rectangle (sert d'unite de separation entre deux calculs d'energie)
                                         int32 nLinesCount,            // Nombre de subdivisions en Y dans le rectangle
										 int32 nLineLength,            // Nombre de subdivisions en X dans le rectangle
										 int32 nSizeX                  // Nombre de subdivisions en X dans la grille
										) const
{
	// On va faire la transformation inverse : on va calculer les influences de sphere definissant l'ellipsoide
	// sur le rectangle transforme par l'inverse de la transformation sphere -> ellipsoide
	
	// Vecteurs de position de la grille deformee
//	const Vector3D vcSphereRectStart  = m_matTransfBaseShapeInv * vcRectStart;  // Position de la base du rectangle
	Vector3D vcRectNormal  = vcRectAxisX ^ vcRectAxisY;

	// Plan sans intersection avec la sphere d'influence ==> rien a faire
	//	     if( Abs( vcSphereRectNormal.Normalized() * vcSphereRectStart) >= k_fInfluenceRange)
	//  <==> if( Abs( vcSphereRectNormal.Normalized() * vcSphereRectStart) >= 1)
	//  <==> if( Abs( vcSphereRectNormal * vcSphereRectStart) >= vcSphereRectNormal.GetLength())
	//  <==> if( Sqr( vcSphereRectNormal * vcSphereRectStart) >= vcSphereRectNormal.GetSqrLength())
	LengthType rRectNormalLengthSqr = vcRectNormal.GetSquareLength();
	LengthType rRectNormalDotStart  = vcRectNormal * vcRectStart;
	
	if( rRectNormalDotStart * rRectNormalDotStart >= rRectNormalLengthSqr )
		return;

	// Vecteurs de base de la grille deformee (MtransfSphereInv * (stepX,0,0)) et (MtransfSphereInv * (0,stepY,0))
//	const Vector3D vcSphereRectStepX  = m_matTransfBaseShapeInv.GetColumn(0) * vcSteps.x;
//	const Vector3D vcSphereRectStepY  = m_matTransfBaseShapeInv.GetColumn(1) * vcSteps.y;

	// Calcul des valeurs 
//	float fRectProdStepXYLengthSqr = fRectNormalLengthSqr * Sqr(vcSteps.x * vcSteps.y);

	LengthType rQuad1A = rRectNormalLengthSqr;
	LengthType rQuadHalfInv1A = .5f / rQuad1A;
	LengthType rQuad1B = 2.f * (   vcRectAxisX.GetSquareLength() * (vcRectStart * vcRectAxisY)
				             - (vcRectStart * vcRectAxisX) * (vcRectAxisX * vcRectAxisY) );
	LengthType rQuad1C = ( (vcRectStart^vcRectAxisX).GetSquareLength() ) - vcRectAxisX.GetSquareLength();

	LengthType rQuad1Delta = rQuad1B * rQuad1B - (4.f * rQuad1A * rQuad1C) ;

	// Grace au test de rejet precedent, on ne doit jamais avoir delta < 0, sauf erreur d'arrondi des float!
	// Auquel cas, les resultats seront abherrants ( sqrt(delta) -> NaN). 
	// Pour eviter cela, on prend la valeur absolue
	assert( rQuad1Delta > -1E-5 );
	rQuad1Delta = Abs(rQuad1Delta);

	LengthType rQuad1DeltaSqrt  = Sqrt( rQuad1Delta );
	float fFirstY = float ( (-rQuad1B - rQuad1DeltaSqrt) * rQuadHalfInv1A );
	float fLastY  = float ( (-rQuad1B + rQuad1DeltaSqrt) * rQuadHalfInv1A );
	int32 nFirstY = CeilFast(fFirstY);
	int32 nLastY  = FloorFast(fLastY);
	// On reste a l'interieure de la grille (Y positif)
	if( nFirstY < 0)
		nFirstY = 0;
	// On reste a l'interieure de la grille (Y pas trop grand)
	if( nLastY >= nLinesCount)
		nLastY =  nLinesCount-1;

	// Intervalle entierement dehors ou alors entre deux valeurs entieres -> rien a visiter
	if( nFirstY > nLastY)
		return;

	Vector3D vcCurLineStart = vcRectStart + (vcRectAxisY * float(nFirstY) );
	int32 iCurEnergyLine = nFirstY * nSizeX;

	LengthType rQuadraticA         = vcRectAxisX.GetSquareLength();
	LengthType rQuadraticAHalfInv  = 1.f / (2.f * rQuadraticA);

	for( int32 j = nFirstY; j <= nLastY; ++j, iCurEnergyLine += nSizeX, vcCurLineStart += vcRectAxisY)
	{
		LengthType rQuadraticB = 2.f * (vcCurLineStart * vcRectAxisX);
//		LengthType rQuadraticC = vcCurLineStart.GetSquareLength() - m_rInfluenceRangeSqr;
		LengthType rQuadraticC = vcCurLineStart.GetSquareLength() - 1.f;
		LengthType rQuadraticDelta = rQuadraticB * rQuadraticB - 4.f * rQuadraticA * rQuadraticC;

		// Avec les calculs precedant la boucle, a chaque iteration il y aura une ligne de valeurs a calculer
		// On ne doit jamais avoir delta < 0, sauf erreur d'arrondi des float!
		// Auquel cas, les resultats seront abherrants ( sqrt(delta) -> NaN). 
		// Pour eviter cela, on prend la valeur absolue
		assert( rQuadraticDelta > -1E-5f);
		rQuadraticDelta = Abs(rQuadraticDelta);
		
		LengthType rDeltaSqrt   = Sqrt( rQuadraticDelta );
		float fFirstInters = float( (-rQuadraticB - rDeltaSqrt) * rQuadraticAHalfInv );
		float fLastInters  = float( (-rQuadraticB + rDeltaSqrt) * rQuadraticAHalfInv );

		int32 nFirstInters = CeilFast(fFirstInters);
		int32 nLastInters  = FloorFast(fLastInters);
		
		// Si une partie de la sphere est en dehors du rectangle (cas ou la sphere depasse de la grille)
		// on se rabbat sur la partie du rectangle
		if( nFirstInters < 0 ) 
			nFirstInters = 0;
		// TODO : A verifer : > ou >= ?
		if( nLastInters >= nLineLength )
			nLastInters =  nLineLength-1;

		// Si les intersections sont entre deux points, l valeur de ces derniers sera bien sur 0
		// De plus, si nFirstInters depasse a droite ou nLastInters a gauche, inutile d'aller plus loin
		if( nFirstInters <= nLastInters )
		{
			// Les performances sont ici importantes
			//      Vector3D vcCur = vcCurLineStart + (vcSphereRectStepX * float(nFirstInters) );

//			Vector3D  vcFirstLinePoint = vcCurLineStart + (vcRectAxisX * fFirstIntersIdx);
			Vector3D vcFirstLinePoint = vcRectAxisX;
			vcFirstLinePoint *= LengthType(nFirstInters);
			vcFirstLinePoint += vcCurLineStart;

			float* pafCurEnergy = &( pafAddEnergies[iCurEnergyLine+ nFirstInters]);
			int32 nSteps = nLastInters-nFirstInters+1;
			AddLineEnergy( pafCurEnergy, nSteps, vcFirstLinePoint, vcRectAxisX);
		}

	}
}



inline void BlobEllipsoid::AddLineEnergy(float* pafCurEnergy, int32 nSteps, const Vector3D& vcFirst, const Vector3D& vcStep) const
{
#if 0
	float* pCurEnergy = pafCurEnergy;
	Vector3D vcCur = vcFirst;
	for( int32 i = nSteps-1; i > 0; --i)
	{
		float fRadDivInflSqr = GetBaseShapeSqrDist(vcCur);
		*(pCurEnergy++) += EnergyNotZeroAtSqrDist(fRadDivInflSqr);
		vcCur += vcStep;
	}
#else
	switch( GetStyle() )
	{
	case etQuadratic: AddLineEnergyConfig <int(etQuadratic)>(pafCurEnergy, nSteps, vcFirst, vcStep); break;
	case etPolynom4 : AddLineEnergyConfig <int(etPolynom4) >(pafCurEnergy, nSteps, vcFirst, vcStep); break;
	case etPolynom6 : AddLineEnergyConfig <int(etPolynom6) >(pafCurEnergy, nSteps, vcFirst, vcStep); break;
	default: assert(0); break;
	}
#endif
}



template <int EVALTYPE>
inline void BlobEllipsoid::AddLineEnergyConfig(float* pafCurEnergy, int32 nSteps, const Vector3D& vcFirst, const Vector3D& vcStep) const
{
	float* pCurEnergy = pafCurEnergy;
	float fP2 = float( vcFirst.GetSquareLength() );
	float fV2 = float( vcStep.GetSquareLength() );
	float f2PV = 2.f * float(vcFirst * vcStep);

	for( int32 i = 0; i < nSteps; ++i)
	{
		float fVal = float(i);
		float fSqrDist = fP2 + fVal* (f2PV + fVal * fV2);
		// Le point courant doit etre dans la sphere de rayon 1
		assert( fSqrDist < 1.001f );
		float fEnergy = EnergyNotZeroGetAtSqrDistConfig<EVALTYPE> (fSqrDist);
		*(pCurEnergy++) += fEnergy;
	}
}


} // namespace



